#ifndef CONTEXT_MENU_H
#define CONTEXT_MENU_H

#include <QWidget>
#include <QListWidget>
#include <QLabel>
#include <QSet>
#include <QMap>
#include <QStringList>
#include <QPushButton>
#include <QMutex>
#include <QFile>
#include <QTimer>
#include <QRegularExpression>
#include <functional> // [NEW] Required for std::function in createActionButton

// Struct to hold regex mapping rules
struct NameMappingRule {
    QRegularExpression regex;
    QString targetName;
};

class ContextMenuWidget : public QWidget
{
    Q_OBJECT

public:
    explicit ContextMenuWidget(QWidget *parent = nullptr);

protected:
    void showEvent(QShowEvent *event) override;

private slots:
    void onSelectFolderClicked();
    void onRestoreAllClicked();
    void onBlockListClicked();
    void processRestoreQueue();

private:
    // --- UI Setup & Helpers ---
    void setupUI();
    void refreshList();

    // Factory method to reduce redundancy in setupUI
    QPushButton* createActionButton(const QString &text, const QString &tooltip, const QString &bgColor, std::function<void()> onClick);

    // --- Core Logic Helpers ---
    void loadRules();
    // [REMOVED] setupLogging(); -> Now uses global logger from winxy.cpp

    void recoverAccidentalDisabledItems();
    void updateStatusLabel();
    void updateDescriptionLabel();
    void refreshSingleItem(QPushButton *btn, const QString &itemName);
    bool archiveItemToDisabled(const QString &regPath, const QString &itemName);

    // entralized Registry Command Helper
    bool runRegCommand(const QString &op, const QString &path, const QString &extraArg = QString());

    // --- Path Finding ---
    void findDisabledFolder();

    // --- Scanning Logic ---
    void scanActiveRegistry();
    void scanDisabledItems();
    void scanDisabledFolder(QMap<QString, QStringList> &targetMap);

    void scanShellExHandlers(const QString &path);
    void scanShellVerbs(const QString &path);

    // --- Utility ---
    QString resolveClsidName(const QString &clsidOrName);
    QString cleanName(const QString &rawName);
    QString sanitizeFileName(const QString &regPath);
    bool isSystemEntry(const QString &friendlyName, const QString &keyName);
    bool checkAdminRights();

    // --- UI State ---
    void updateButtonStyle(QPushButton *btn, bool isActive, bool isDisabled);
    QPushButton* findButtonByName(const QString &itemName);

    // --- Member Variables ---
    QListWidget *listWidget;
    QLabel *statusLabel;
    QLabel *m_descLabel;

    QPushButton *btnSelectFolder;
    QPushButton *btnRestoreAll;
    QPushButton *btnBlockList;

    QString m_disabledDir;

    // Data Storage
    QMap<QString, QStringList> m_activePaths;
    QMap<QString, QStringList> m_disabledPaths;

    // Loaded Rules
    QString m_jsonDisabledPath;
    QString m_removeRegexPattern;
    QList<NameMappingRule> m_nameMappingRules;
    QStringList m_forceKeepKeywords;
    QStringList m_skipKeywords;
    QStringList m_exactSkipList;

    // Animation / Bulk Operation State
    QTimer *m_restoreTimer;
    QStringList m_restoreQueue;

    bool m_isAdmin;
    bool m_isPathValid;
};

#endif // CONTEXT_MENU_H
